/**
* \file: AlsaAudioSinkImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*
* \copyright (c) 2014-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <sys/prctl.h>
#include <adit_logging.h>
#include "AlsaAudioSinkImpl.h"

#include <inttypes.h>

LOG_IMPORT_CONTEXT(aauto_audio)

namespace adit { namespace aauto {
// TODO thread-safety
// TODO improve configuration stuff
// TODO remove superfluous AlsaAudioIntf
AlsaAudioSinkImpl::AlsaAudioSinkImpl(AudioSink* inSink)
{
    mInitialized = false;
    mStarted = false;
    mSessionId = -1;
    mQueueRunning = false;
    mShutdown = false;

    audioSink = inSink;
    mCallbacks = nullptr;
    mIsSetThreadParam = false;

    AudioSinkRecordRunning = false;
}

AlsaAudioSinkImpl::~AlsaAudioSinkImpl()
{
    if (!mShutdown)
    {
        shutdown();
    }
}

bool AlsaAudioSinkImpl::init()
{
    // TODO if use after shutdown is not allowed then block it
    if (mInitialized)
    {
        LOG_ERROR((aauto_audio, "%s is already initialized!", sinkName(mConfig.mStreamType)));
        return false;
    }

    /* read and set configuration which was set by Application */
    if (true != getCurrConfig()) {
        LOG_ERROR((aauto_audio, "%s, init()  getCurrConfig() failed", sinkName(mConfig.mStreamType)));
        return false;
    }

    /* register callbacks to GalReceiver */
    audioSink->registerCallbacks(this);
    /* set GalReceiver::AudioSink configuration */
    audioSink->setMaxUnackedFrames(mConfig.mUnackedFrames);
    audioSink->setCodecType(mConfig.mCodec);
    audioSink->setAudioType(mConfig.mStreamType);
    (void)audioSink->addSupportedConfiguration(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);

    /* create AlsaAudio object for playback */
    mPlayer = new AlsaAudioCommon(mConfig.mAlsaLogging, mConfig.mVerbose);
    if (mPlayer == nullptr)
    {
        LOG_ERROR((aauto_audio, "%s, error cannot create AlsaAudioCommon(sink) class", sinkName(mConfig.mStreamType)));
        return false;
    }
    else
    {
        LOGD_DEBUG((aauto_audio, "%s, create AlsaAudioCommon(sink) class successfully", sinkName(mConfig.mStreamType)));
    }

    mInitialized = true;
    mShutdown = false;
    return true;
}

void AlsaAudioSinkImpl::shutdown()
{
    mInitialized = false;
    mShutdown = true;

    // in case we where still running stop and notify
    if (mStarted)
    {
        stop(true /* in case of shutdown, discard remaining work items */);
        if (mCallbacks != nullptr)
            mCallbacks->playbackStopCallback(mSessionId);

        LOGD_DEBUG((aauto_audio, "%s, id=%d is down", sinkName(mConfig.mStreamType), mSessionId));
    }

    // delete audio class
    mPlayer = nullptr;

    if (AudioSinkRecordRunning) {
        mAudioSinkRecord.close();
        AudioSinkRecordRunning = false;
    }
}

bool AlsaAudioSinkImpl::getCurrConfig(void)
{
    // read out configuration parameters
    if (true != mConfig.ResultConfig())
    {
        LOG_ERROR((aauto_audio, "getCurrConfig()  ResultConfig() failed"));
        return false;
    }

    if (mConfig.mConfSType == "AUDIO_STREAM_MEDIA")
    {
        mConfig.mStreamType = AUDIO_STREAM_MEDIA;
    }
    else if (mConfig.mConfSType == "AUDIO_STREAM_SYSTEM_AUDIO")
    {
        mConfig.mStreamType = AUDIO_STREAM_SYSTEM_AUDIO;
    }
    else if (mConfig.mConfSType == "AUDIO_STREAM_GUIDANCE")
    {
        mConfig.mStreamType = AUDIO_STREAM_GUIDANCE;
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, not support stream type.", sinkName(mConfig.mStreamType)));
        return false;
    }

    if (mConfig.mConfCodec == "MEDIA_CODEC_AUDIO_PCM")
    { 
        /* supported */
        mConfig.mCodec = MEDIA_CODEC_AUDIO_PCM;
    }
    else if (mConfig.mConfCodec == "MEDIA_CODEC_AUDIO_AAC_LC_ADTS")
    {
        mConfig.mCodec = MEDIA_CODEC_AUDIO_AAC_LC_ADTS;
        LOG_ERROR((aauto_audio, "%s, not support codec type.", sinkName(mConfig.mStreamType)));
        return false;
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Unknown codec type.", sinkName(mConfig.mStreamType)));
        return false;
    }

    if ((mConfig.mStreamType == AUDIO_STREAM_MEDIA) && 
        (mConfig.mSampleRate == MEDIA_AUDIO_SAMPLING_RATE) && 
        (mConfig.mNumBits == MEDIA_AUDIO_BITS_PER_SAMPLE) && 
        (mConfig.mNumChannels == MEDIA_AUDIO_CHANNELS))
    {
        /* ok */
    }
    else if ((mConfig.mStreamType == AUDIO_STREAM_SYSTEM_AUDIO) && 
            (mConfig.mSampleRate == SYSTEM_AUDIO_SAMPLING_RATE) && 
            (mConfig.mNumBits == SYSTEM_AUDIO_BITS_PER_SAMPLE) && 
            (mConfig.mNumChannels == SYSTEM_AUDIO_CHANNELS))
    {
        /* ok */
    }
    else if ((mConfig.mStreamType == AUDIO_STREAM_GUIDANCE) && 
            (mConfig.mSampleRate == GUIDANCE_AUDIO_SAMPLING_RATE) && 
            (mConfig.mNumBits == GUIDANCE_AUDIO_BITS_PER_SAMPLE) && 
            (mConfig.mNumChannels == GUIDANCE_AUDIO_CHANNELS))
    {
        /* ok */
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Error setting parameter sampling-rate, bits-per-sample, channels",
                sinkName(mConfig.mStreamType)));
        return false;
    }

    if (mConfig.mUnackedFrames <= 0)
    {
        LOG_ERROR((aauto_audio, "%s, Error setting parameter audio-sink-max-unacked-frames",
                sinkName(mConfig.mStreamType)));
        return false;
    }

    return true;
}

void AlsaAudioSinkImpl::setConfigItem(string inKey, string inValue)
{
    LOGD_DEBUG((aauto_audio, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(),inValue.c_str())); 
    mConfig.set(inKey, inValue);
}

void AlsaAudioSinkImpl::registerCallbacks(IAditAudioSinkCallbacks* inCallbacks)
{
    mCallbacks = inCallbacks;
}

/* IAudioSinkCallbacks */
void AlsaAudioSinkImpl::dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
        uint8_t* data, size_t len)
{
    LOG_FATAL((aauto_audio, "%s, non-zero copy version of dataAvailableCallback is not supported!",
            sinkName(mConfig.mStreamType)));
}

void AlsaAudioSinkImpl::dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            const shared_ptr<IoBuffer>& galFrame, uint8_t* data, size_t len)
{
    if (mQueueRunning)
    {
        /* create new workqueue item for received data */
        shared_ptr<WorkItem> work(new AudioPlaybackWorkItem(this, sessionId, timestamp,
                galFrame, data, len));
        /* queue item which played out by workqueue thread */
        mWorkQueue->queueWork(work, (unsigned int)len);
        if (mConfig.mVerbose)
        {
            LOGD_VERBOSE((aauto_audio, "%s, id=%d has data available (len=%zu) timestamp=%" PRIu64 "",
                    sinkName(mConfig.mStreamType), mSessionId, len, timestamp));
        }
        /* record received AudioSink data into WAV file
         * in case Application enables the feature */
        if ((mConfig.mAudioRecord) && (AudioSinkRecordRunning)) {
            int res = mAudioSinkRecord.write(data, (unsigned int)len);
            if (res != (int)len) {
                LOG_ERROR((aauto_audio, "%s write data for AudioSinkRecord failed=%d",
                        sinkName(mConfig.mStreamType), res));
            }
        }
    }
    else
    {
        LOG_WARN((aauto_audio, "%s, id=%d is not ready for data yet",
                sinkName(mConfig.mStreamType), mSessionId));
    }
}

int AlsaAudioSinkImpl::codecConfigCallback(uint8_t* data, size_t len)
{
    /* Invoked when there is new codec configuration data.
     * Currently, only PCM is supported */
    LOGD_DEBUG((aauto_audio, "%s codec config callback", sinkName(mConfig.mStreamType)));
    return STATUS_SUCCESS;
}

int AlsaAudioSinkImpl::setupCallback(int mediaCodecType)
{
    // setup each mediaCodecType. like ALSA prepare. this AlsaAudioSink implement was unnecessary.
    LOGD_DEBUG((aauto_audio, "%s setup callback", sinkName(mConfig.mStreamType)));
    return STATUS_SUCCESS;
}

void AlsaAudioSinkImpl::playbackStartCallback(int32_t inSessionId)
{
    LOGD_DEBUG((aauto_audio, "%s, playbackStartCallback() id=%d",
            sinkName(mConfig.mStreamType), inSessionId));

    if (mStarted)
    {
        LOG_ERROR((aauto_audio, "%s, id=%d already got playback start!",
                sinkName(mConfig.mStreamType), mSessionId));
        return;
    }

    mStarted = true;
    bool callbackOK = true;
    mSessionId = inSessionId;

    /* null callbacks are OK */
    if (mCallbacks != nullptr)
    {
        /* notify Application to start audio capture */
        mCallbacks->playbackStartCallback(inSessionId);
        callbackOK = true; // TODO add return code to playbackStartCallback
    }

    if (callbackOK)
    {
        /* read and set configuration again because Application
         * got set new configuration values due to playbackStartCallback */
        if (true != getCurrConfig()) {
            LOG_ERROR((aauto_audio, "%s, playbackStartCallback()  getCurrConfig() failed", sinkName(mConfig.mStreamType)));

            if (mCallbacks != nullptr)
            {
                mCallbacks->notifyErrorCallback(AUDIO_SINK_CONFIGURATION_ERROR);
            }
        }
        audioSink->setMaxUnackedFrames(mConfig.mUnackedFrames);
        audioSink->setCodecType(mConfig.mCodec);
        (void)audioSink->addSupportedConfiguration(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);
        audioSink->setAudioType(mConfig.mStreamType);

        /* start audio playback */
        bool result = start();
        if (result == false)
        {
            if (mCallbacks != nullptr)
            {
                mCallbacks->notifyErrorCallback(AUDIO_SINK_START_ERROR);
            }
        }
    }

    /* open the WAV file to store the AudioSink record */
    if (mConfig.mAudioRecord) {
        if (0 != mAudioSinkRecord.open(mConfig.mAudioRecordFile , sinkName(mConfig.mStreamType))) {
            LOG_ERROR((aauto_audio, "%s, playbackStartCallback() Open AudioSinkRecord file failed.",
                    sinkName(mConfig.mStreamType)));
        } else {
            AudioSinkRecordRunning = true;
        }
    }
}

void AlsaAudioSinkImpl::playbackStopCallback(int32_t inSessionId)
{
    LOGD_DEBUG((aauto_audio, "%s, playbackStopCallback() id=%d", sinkName(mConfig.mStreamType), inSessionId));

    if (!mStarted)
    {
        LOG_ERROR((aauto_audio, "%s, got stop without start, id=%d!",
                sinkName(mConfig.mStreamType), inSessionId));
        // continue anyway
    }
    if (inSessionId != mSessionId)
    {
        LOG_ERROR((aauto_audio, "%s, %id got stop with different id=%d",
                sinkName(mConfig.mStreamType), mSessionId, inSessionId));
        // continue anyway
    }

    /* stop audio playback */
    mStarted = false;
    /* discard remaining work items was configured by Application.
     * By default it's set to false. */
    stop(mConfig.mDiscardFrames);

    /* null callbacks are OK */
    if (mCallbacks != nullptr)
    {
        /* notify Application to stop audio playback */
        mCallbacks->playbackStopCallback(inSessionId);
    }

    mSessionId = -1; // reset session id

    /* insert WAV header and close the AudioSink record*/
    if ((mConfig.mAudioRecord) && (AudioSinkRecordRunning)) {
        if (0 != mAudioSinkRecord.createWavHeader(mConfig.mCodec, mConfig.mNumChannels, mConfig.mNumBits, mConfig.mSampleRate)) {
            LOG_ERROR((aauto_audio, "%s, playbackStopCallback() create wav header for AudioSinkRecord failed.",
                        sinkName(mConfig.mStreamType)));
        }
        mAudioSinkRecord.close();
    }
}


/* WorkItem */
void AlsaAudioSinkImpl::AudioPlaybackWorkItem::SetThreadParam(const char* threadName)
{
    /* set thread name */
    string tmpSinkName = mSink->sinkName(mSink->mConfig.mStreamType);
    string sName = threadName + tmpSinkName; //too long name, less than 15 chars ??
    prctl(PR_SET_NAME, sName.c_str(), 0, 0, 0);

    if(!mSink->mConfig.mDisablePrio)
    {
        /* set thread priority */
        int err = 0;
        int schedPolicy;
        struct sched_param schedParam;

        err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
        if(err == 0)
        {
            schedParam.sched_priority = mSink->mConfig.mThreadPrio;
            err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
            if(err != 0)
            {
                LOG_ERROR((aauto_audio, "%s: set priority failed with error %d", sName.c_str(), err));
            }
        }
        else
        {
            LOG_ERROR((aauto_audio, "%s: get priority failed with error %d", sName.c_str(), err));
        }
    }
}

void AlsaAudioSinkImpl::AudioPlaybackWorkItem::run()
{
    if (!mSink->mIsSetThreadParam)
    {
        SetThreadParam("AlsaAudioSink");
        mSink->mIsSetThreadParam = true; /* setting only once */
    }

    if (mSink->mConfig.mVerbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, run() DataPtr=%p, Len=%zu, Timestamp=%" PRIu64 "",
                mSink->sinkName(mSink->mConfig.mStreamType), mDataPtr, mLen, mTimestamp));
    }

    int err = 0;
    /* write queued audio data to AlsaAudio object (ALSA device) */
    if ((err = mSink->mPlayer->ReadWrite(mDataPtr, mLen)) < 0)
    {
        // TODO we should inform Application about error,
        //      but there is no possibility at the moment
        LOG_ERROR((aauto_audio, "%s, run() write to ALSA in work queue failed with %d",
                mSink->sinkName(mSink->mConfig.mStreamType), err));
        if (mSink->mCallbacks != nullptr)
        {
            mSink->mCallbacks->notifyErrorCallback(AUDIO_SINK_WRITE_ERROR);
        }
    }
    else
    {
        /* send ACK to MD that audio data has been processed */
        mSink->audioSink->ackFrames(mSessionId, 1);
    }
}

/* private methods */
bool AlsaAudioSinkImpl::start()
{
    unsigned int samples = 0;

    /* set-up ALSA device for playback */
    if (prepare())
    {
        LOGD_DEBUG((aauto_audio, "%s, ALSA prepare successfully", sinkName(mConfig.mStreamType)));
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Error ALSA prepare", sinkName(mConfig.mStreamType)));
        return false;
    }

    // check if threshold was configured by Application
    if (mConfig.mThresholdMs > 0) {
        switch (mConfig.mSampleRate) {
            case MEDIA_AUDIO_SAMPLING_RATE:
            {
                samples = SAMPLES_IN_ONE_FRAME_FOR_48KHZ;
                break;
            }
            case GUIDANCE_AUDIO_SAMPLING_RATE:
            // case SYSTEM_AUDIO_SAMPLING_RATE:
            {
                if (GUIDANCE_AUDIO_SAMPLING_RATE != SYSTEM_AUDIO_SAMPLING_RATE) {
                    LOG_WARN((aauto_audio, "%s: Different sample rate for Guidance(%d) and SystemAudio(%d) not handled!",
                            sinkName(mConfig.mStreamType), GUIDANCE_AUDIO_SAMPLING_RATE, SYSTEM_AUDIO_SAMPLING_RATE));
                }
                samples = SAMPLES_IN_ONE_FRAME_FOR_16KHZ;
                break;
            }
            default:
            {
                LOG_ERROR((aauto_audio, "%s: Unknown sample rate %d. 16kHz and 48kHz are supported.",
                        sinkName(mConfig.mStreamType), mConfig.mSampleRate));
                samples = 0;
                break;
            }
        };
    } else {
        // TODO: Debug or Verbose log level?
        LOGD_DEBUG((aauto_audio, "%s: No threshold defined. No pre-buffering.", sinkName(mConfig.mStreamType)));
    }

    /* create workqueue and set threshold - if configured,
     * workqueue thread starts after threshold was reached */
    mWorkQueue = new WorkQueue(1, false, mPlayer->GetThresholdBufferSize(samples));
    if (!mWorkQueue->start())
    {
        LOG_ERROR((aauto_audio, "%s, id=%d could not start work queue",
                sinkName(mConfig.mStreamType), mSessionId));
        return false;
    }

    /* set thread parameter */
    mIsSetThreadParam = false;

    mQueueRunning = true;
    return true;
}

void AlsaAudioSinkImpl::stop(bool discardRemaining)
{
    if (mPlayer != nullptr) {
        /* stop processing before shutdown workqueue */
        if (mShutdown) {
            mPlayer->setAlsaAudioState(mPlayer->AlsaAudioStates::AlsaAudioStop);
            LOGD_DEBUG((aauto_audio, "%s, stop() State set to AlsaAudioStop due to shutdown",
                      sinkName(mConfig.mStreamType)));
        } else if (discardRemaining) {
            /* handle current work item and stop afterwards */
            mPlayer->setAlsaAudioState(mPlayer->AlsaAudioStates::AlsaAudioDrop);
        } else {
            /* handle all queued work items and stop afterwards */
            mPlayer->setAlsaAudioState(mPlayer->AlsaAudioStates::AlsaAudioDrain);
        }
    }

    mQueueRunning = false;

    /* close audio sink thread(s) and wait for closure */
    if (mWorkQueue != nullptr) {
        if (discardRemaining) {
            /* play current item and discard all other queued items from workqueue */
            mWorkQueue->shutdown();

            // send ACK to MD for all queued items which will be discarded
            uint32_t itemsQueued = mWorkQueue->getNumOfWorkItems();
            // we can use directly the number of queued items as the number of frames to be acknowledged
            if (itemsQueued > 0) {
                audioSink->ackFrames(mSessionId, itemsQueued);
                LOGD_DEBUG((aauto_audio, "%s, id=%d Sent ACK for %u queued and discarded items",
                        sinkName(mConfig.mStreamType), mSessionId, itemsQueued));
            }
        } else {
            LOGD_DEBUG((aauto_audio, "%s, id=%d shutting down work queue by waitShutdown()",
                    sinkName(mConfig.mStreamType), mSessionId));
            /* play all queued items out before shutdown */
            mWorkQueue->waitShutdown();
        }
    }
    mWorkQueue = nullptr;

    /* stop audio playback */
    if (mPlayer != nullptr)
    {
        /* stop streaming and wait for all
         * pending frames in ALSA to be played */
        mPlayer->StopStreaming();
        /* close ALSA device */
        mPlayer->ReleasePCM();
    }

    LOGD_DEBUG((aauto_audio, "%s, id=%d stopped", sinkName(mConfig.mStreamType), mSessionId));
}

/* Preparations of ALSA */
bool AlsaAudioSinkImpl::prepare()
{
    LOGD_DEBUG((aauto_audio, "%s, audio stream start(sample rate:%d, bits per channel:%d, channels:%d)",
            sinkName(mConfig.mStreamType), mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels));

    /* set configuration values for ALSA device */
    mPlayer->SetStreamName(sinkName(mConfig.mStreamType));
    mPlayer->SetDir( SND_PCM_STREAM_PLAYBACK );
    mPlayer->SetRate( mConfig.mSampleRate );
    mPlayer->SetInitTout(mConfig.mInitToutms);
    mPlayer->SetChannels( mConfig.mNumChannels );
    mPlayer->SetIdealPeriodMs( mConfig.mPeriodms );
    if (mConfig.mBufferperiods >= 0)
    {
        mPlayer->SetBufferIdealPeriods( mConfig.mBufferperiods );
    }
    if (mConfig.mSilencems >= 0)
    {
        mPlayer->SetPrefillMs ( mConfig.mSilencems );
    }
    mPlayer->SetFormat( SND_PCM_FORMAT_S16_LE );

    mPlayer->SetThresholdMs(mConfig.mThresholdMs);

    /* set up ALSA device */
    if (0 != mPlayer->SetupPCM((char*)mConfig.mDevice.c_str()))
    {
        return false;
    }

    LOGD_DEBUG((aauto_audio, "%s, device=%s, period=%dms",
            sinkName(mConfig.mStreamType), mConfig.mDevice.c_str(), mConfig.mPeriodms));

    return true;
}

} } /* namespace adit { namespace aauto */
